import torch

from attacks import Attack
import torch.nn.functional as F

from constants import DEVICE
from utils import cross_entropy_loss, de_normalization, normalization


class FIEPNIFGSM(Attack):
    """ FIEPNIFGSM: FI + EPNI-FGSM """

    def __init__(self, model, eps=16 / 255, steps=10, decay=1.0, epochs=5, drop_probability=0.3, ensemble_number=30,
                 intermediate_layer_name='Mixed_5b'):
        """
        :param model: DNN model
        :param eps: the maximum perturbation
        :param steps: the number of iterations
        :param decay: the decay factor
        :param drop_probability: the drop probability
        :param ensemble_number: the number of random masks
        :param intermediate_layer_name: the name of k-th layer   feature maps
        """
        super().__init__("FIEPNIFGSM", model)
        self.eps = eps
        self.steps = steps
        self.alpha = self.eps / self.steps
        self.decay = decay
        self.epochs = epochs
        self.drop_probability = drop_probability
        self.ensemble_number = ensemble_number
        self.intermediate_layer_name = intermediate_layer_name
        self.intermediate_layer_feature_maps = None
        self.register_hook()

    def hook(self, module, input, output):
        self.intermediate_layer_feature_maps = output
        return None

    def register_hook(self):
        for name, module in self.model.named_modules():
            if name == self.intermediate_layer_name:
                module.register_forward_hook(hook=self.hook)

    def get_aggregate_gradient(self, images, targets):
        _ = self.model(images)
        aggregate_grad = torch.zeros_like(self.intermediate_layer_feature_maps)
        for _ in range(self.ensemble_number):
            mask = torch.bernoulli(torch.full_like(images, 1 - self.drop_probability))
            images_masked = images * mask
            logits = self.model(images_masked)
            loss = cross_entropy_loss(logits, targets)
            aggregate_grad += torch.autograd.grad(loss, self.intermediate_layer_feature_maps)[0]
        aggregate_grad /= torch.sqrt(torch.sum(torch.square(aggregate_grad), dim=(1, 2, 3), keepdim=True))
        return aggregate_grad

    def FIA_loss_function(self, aggregate_grad, x):
        _ = self.model(x)
        # FIA_loss = torch.sum(aggregate_grad * self.intermediate_layer_feature_maps) / torch.numel(
        #     self.intermediate_layer_feature_maps)
        FIA_loss = torch.sum(aggregate_grad * self.intermediate_layer_feature_maps)
        return FIA_loss

    def forward(self, images, labels):
        targets = F.one_hot(labels.type(torch.int64), 1000).float().to(DEVICE)
        images_de_normalized = de_normalization(images)
        images_min = torch.clamp(images_de_normalized - self.eps, min=0.0, max=1.0)
        images_max = torch.clamp(images_de_normalized + self.eps, min=0.0, max=1.0)

        aggregate_grad = self.get_aggregate_gradient(images, targets)
        g = torch.zeros_like(images)
        for _ in range(self.epochs):
            adv_hat = images.clone()
            for _ in range(self.steps):
                adv_hat_exchanged = torch.stack([x[torch.randperm(3), :, :] for x in adv_hat])

                loss_hat = self.FIA_loss_function(aggregate_grad, adv_hat_exchanged)
                grad_hat = torch.autograd.grad(loss_hat, adv_hat)[0]

                adv_hat_de_normalized = de_normalization(adv_hat)
                adv_wave_de_normalized = adv_hat_de_normalized + self.alpha * grad_hat / torch.mean(torch.abs(grad_hat),
                                                                                                    dim=(1, 2, 3),
                                                                                                    keepdim=True) + self.decay * self.alpha * g
                adv_wave = normalization(adv_wave_de_normalized)

                loss_wave = self.FIA_loss_function(aggregate_grad, adv_wave)
                grad_wave = torch.autograd.grad(loss_wave, adv_wave)[0]

                g = self.decay * g + grad_hat / torch.mean(torch.abs(grad_hat), dim=(1, 2, 3),
                                                           keepdim=True) + grad_wave / torch.mean(torch.abs(grad_wave),
                                                                                                  dim=(1, 2, 3),
                                                                                                  keepdim=True)

                adv_hat_de_normalized = de_normalization(adv_hat)
                adv_hat_de_normalized = torch.clamp(adv_hat_de_normalized + self.alpha * torch.sign(g), min=images_min,
                                                    max=images_max)
                adv_hat = normalization(adv_hat_de_normalized)

        adv = images.clone()
        for _ in range(self.steps):
            loss = self.FIA_loss_function(aggregate_grad, adv)
            grad = torch.autograd.grad(loss, adv)[0]

            adv_de_normalized = de_normalization(adv)
            adv_de_normalized = adv_de_normalized + self.alpha * grad / torch.mean(torch.abs(grad), dim=(1, 2, 3),
                                                                                   keepdim=True) + self.decay * self.alpha * g
            pre_adv = normalization(adv_de_normalized)

            pre_loss = self.FIA_loss_function(aggregate_grad, pre_adv)
            pre_grad = torch.autograd.grad(pre_loss, pre_adv)[0]

            g = self.decay * g + grad / torch.mean(torch.abs(grad), dim=(1, 2, 3),
                                                   keepdim=True) + pre_grad / torch.mean(torch.abs(pre_grad),
                                                                                         dim=(1, 2, 3), keepdim=True)

            adv_de_normalized = de_normalization(adv)
            adv_de_normalized = torch.clamp(adv_de_normalized + self.alpha * torch.sign(g), min=images_min,
                                            max=images_max)
            adv = normalization(adv_de_normalized)

        return adv
